package com.ejie.ab04b.util;

import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.ejie.ab04b.constantes.Constantes;
import com.ejie.ab04b.constantes.ConstantesPlateaTramitacion;
import com.ejie.ab04b.model.ComunicacionApertura;
import com.ejie.ab04b.model.OS2;
import com.ejie.ab04b.model.OS3;
import com.ejie.ab04b.parse.MessageUnparser;
import com.ejie.detailinfo.aperturassoltel.xml.GrupoDatosEmpresa.CampoRadioEmpresa;
import com.ejie.detailinfo.aperturassoltel.xml.RadioSN;
import com.ejie.detailinfo.aperturassoltel.xml.RadioSiNo;
import com.ejie.x43k.xml.Documento;

/**
 * @author GFI-NORTE
 * 
 */
public final class ToolkitUtils {

	private static ToolkitUtils instance;

	private static final Logger LOGGER = LoggerFactory
			.getLogger(ToolkitUtils.class);

	/**
	 * Constructor privado.
	 */
	private ToolkitUtils() {

	}

	/**
	 * Gets the single instance of ToolkitUtils.
	 * 
	 *  ToolkitParameterUtils
	 *
	 * @return single instance of ToolkitUtils
	 */
	public static ToolkitUtils getInstance() {
		if (ToolkitUtils.instance == null) {
			ToolkitUtils.instance = new ToolkitUtils();
		}
		return ToolkitUtils.instance;
	}

	/**
	 * A partir de un Documento, devuelve su representación XML en un String.
	 * 
	 *  documento
	 *            objeto de tipo Documento
	 *  String e
	 *
	 * @param documento the documento
	 * @return the documento as string
	 */
	public String getDocumentoAsString(Documento documento) {
		MessageUnparser<Documento> unparser = new MessageUnparser<Documento>();
		String resultado = null;
		try {
			resultado = unparser.parse(documento);
		} catch (Exception e) {
			ToolkitUtils.LOGGER.error(
					"Error en el parseo del objeto documento", e);
		}
		return resultado;
	}

	/**
	 * Concatenar direccion toolkit.
	 * 
	 *  calle
	 *            String
	 *  portal
	 *            String
	 *  piso
	 *            String
	 *  letra
	 *            String
	 *  String
	 *
	 * @param calle the calle
	 * @param portal the portal
	 * @param piso the piso
	 * @param letra the letra
	 * @return the string
	 */
	public String concatenarDireccionToolkit(String calle, String portal,
			String piso, String letra) {

		StringBuilder sb = new StringBuilder(calle);
		if (!Utilities.getInstance().isEmpty(portal)) {
			sb.append(Constantes.ESPACIO).append(portal);
			if (!Utilities.getInstance().isEmpty(piso)) {
				sb.append(Constantes.COMA).append(Constantes.ESPACIO)
						.append(piso).append(Constantes.GRADO);
				if (!Utilities.getInstance().isEmpty(piso)) {
					sb.append(Constantes.ESPACIO).append(letra);
				}
			}
		}
		return sb.toString();
	}

	/**
	 * Parsear fecha.
	 * 
	 *  sfecha
	 *            String
	 *  Date
	 *
	 * @param sfecha the sfecha
	 * @return the date
	 */
	public Date parsearFecha(String sfecha) {

		Date fecha = null;
		DateFormat df = null;
		try {
			// Intentamos parsear en formato es
			df = new SimpleDateFormat(Constantes.ES_DATE_MASK);
			fecha = df.parse(sfecha);
		} catch (ParseException pe) {
			try {
				// Si falla, intentamos parsear en formato eu
				df = new SimpleDateFormat(Constantes.EU_DATE_MASK);
				fecha = df.parse(sfecha);
			} catch (ParseException pe2) {
				// No hacemos nada, devolvemos null
			}
		}
		return fecha;

	}

	/**
	 * Convertir tipo identificacion.
	 * 
	 *  tipoIdent
	 *            String
	 *  String
	 *
	 * @param tipoIdent the tipo ident
	 * @return the string
	 */
	public String convertirTipoIdentificacion(String tipoIdent) {

		if (ConstantesPlateaTramitacion.EJGV_PERSONTYPE_CIF.equals(tipoIdent)) {
			return Constantes.TIPO_CIF;
		} else if (ConstantesPlateaTramitacion.EJGV_PERSONTYPE_DNI
				.equals(tipoIdent)) {
			return Constantes.TIPO_NIF;
		}
		return Constantes.TIPO_OTROS;
	}

	/**
	 * Convertir radio empresa nueva existente.
	 * 
	 *  radioEmpresa
	 *            CampoRadioEmpresa
	 *  String
	 *
	 * @param radioEmpresa the radio empresa
	 * @return the string
	 */
	public String convertirRadioEmpresaNuevaExistente(
			CampoRadioEmpresa radioEmpresa) {

		if (radioEmpresa.getN() != null) {
			return Constantes.EMPRESA_NUEVA;
		} else if (radioEmpresa.getE() != null) {
			return Constantes.EMPRESA_EXISTENTE;
		}
		return null;
	}

	/**
	 * Convertir radio si no.
	 * 
	 *  radioSiNo
	 *            String
	 *  String
	 *
	 * @param radioSiNo the radio si no
	 * @return the string
	 */
	public String convertirRadioSiNo(String radioSiNo) {

		if (Constantes.TOOLKIT_SI.equals(radioSiNo)) {
			return Constantes.VALUE_SI;
		} else if (Constantes.TOOLKIT_NO.equals(radioSiNo)) {
			return Constantes.VALUE_NO;
		}
		return null;
	}

	/**
	 * Convertir radio si no.
	 * 
	 *  radioSiNo
	 *            RadioSiNo
	 *  String
	 *
	 * @param radioSiNo the radio si no
	 * @return the string
	 */
	public String convertirRadioSiNo(RadioSiNo radioSiNo) {

		if (radioSiNo.getSi() != null) {
			return Constantes.VALUE_SI;
		} else if (radioSiNo.getNo() != null) {
			return Constantes.VALUE_NO;
		}
		return null;
	}

	/**
	 * Convertir radio SN.
	 * 
	 *  radioSN
	 *            RadioSN
	 *  String
	 *
	 * @param radioSN the radio SN
	 * @return the string
	 */
	public String convertirRadioSN(RadioSN radioSN) {

		if (radioSN.getS() != null) {
			return Constantes.VALUE_SI;
		} else if (radioSN.getN() != null) {
			return Constantes.VALUE_NO;
		}
		return null;
	}

	/**
	 * Obtener territorio solicitud apertura.
	 * 
	 *  apertura
	 *            ComunicacionApertura
	 *  String
	 *
	 * @param apertura the apertura
	 * @return the string
	 */
	public String obtenerTerritorioSolicitudApertura(
			ComunicacionApertura apertura) {

		if (Constantes.VALUE_SI.equals(apertura.getIndobra056())) {
			// Es de tipo obra
			return apertura.getObra() != null ? apertura.getObra()
					.getPrvobr082() : "";
		} else {
			// Es de tipo centro de trabajo
			return apertura.getProvinciaCtr().getCodProvincia();
		}
	}

	/**
	 * Obtener territorio solicitud OS 2.
	 * 
	 *  os2
	 *            OS2
	 *  String
	 *
	 * @param os2 the os 2
	 * @return the string
	 */
	public String obtenerTerritorioSolicitudOS2(OS2 os2) {

		return os2.getTeros2060();
	}

	/**
	 * Obtener territorio solicitud OS 3.
	 * 
	 *  os3
	 *            OS3
	 *  String
	 *
	 * @param os3 the os 3
	 * @return the string
	 */
	public String obtenerTerritorioSolicitudOS3(OS3 os3) {

		return os3.getTeros3061();
	}

}
